/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Globalization;

using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Default;
using Borland.Eco.Persistence.Configuration;

namespace Borland.Eco.Persistence.Db2
{
	public sealed class DecimalAsDecimal: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			// DECIMAL wants (precision, scale) 
			// where precision is number of digits, 
			// and scale is number of decimals. 
			// DB2 wants no larger than 31 in size.
			// http://www-3.ibm.com/cgi-bin/db2www/data/db2/udb/winos2unix/support/document.d2w/report?last_page=v8list.d2w&fn=db2v8sqxibmsqlr45.htm
			return String.Format(CultureInfo.InvariantCulture, "DECIMAL({0:d},2)", Math.Min(length, 31)); // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Decimal));
			parameter.DbType = DbType.Decimal ;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToDecimal(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}

	}

	public sealed class Int64AsLongInteger: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "NUMERIC(18,0)"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int64));
			parameter.DbType = DbType.Int64;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt64(value), parameter);
		}


		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Int64)(System.Decimal)columnValue;
		}

	}

	public sealed class Int64AsBigInt: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "BIGINT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Int64));
			parameter.DbType = DbType.Int64;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToInt64(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Int64));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return columnValue;
		}
	}

	public sealed class SingleAsFloat: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "FLOAT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			parameter.DbType = DbType.Double;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToSingle(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Single)(System.Double)columnValue;
		}
	}
}


/*

DB2 Type			BdpType			BdpSubType			System.Type

CHAR			String			stFixed			String
VARCHAR			String						String
SMALLINT			Int16						Int16
BIGINT			Int64						Int32
INTEGER			Int32						Int64
DOUBLE			Double						Double
FLOAT			Float						Single
REAL			Float						Single
DATE			Date						DateTime
TIME			Time						DateTime
TIMESTAMP		Datetime					DateTime
NUMERIC			Decimal						Decimal
DECIMAL			Decimal						Decimal
BLOB			Blob			stBinary			Byte[]
CLOB			Blob			stMemo			Char[]
*/